/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.codec.http2;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.DefaultHttp2HeadersDecoder;
import io.netty.handler.codec.http2.Http2CodecUtil;
import io.netty.handler.codec.http2.Http2Error;
import io.netty.handler.codec.http2.Http2Exception;
import io.netty.handler.codec.http2.Http2Flags;
import io.netty.handler.codec.http2.Http2FrameListener;
import io.netty.handler.codec.http2.Http2FrameReader;
import io.netty.handler.codec.http2.Http2FrameSizePolicy;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2HeadersDecoder;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.util.internal.PlatformDependent;

public class DefaultHttp2FrameReader
implements Http2FrameReader,
Http2FrameReader.Configuration,
Http2FrameSizePolicy {
    private final Http2HeadersDecoder headersDecoder;
    private boolean readingHeaders = true;
    private boolean readError;
    private byte frameType;
    private int streamId;
    private Http2Flags flags;
    private int payloadLength;
    private HeadersContinuation headersContinuation;
    private int maxFrameSize;

    public DefaultHttp2FrameReader() {
        this(true);
    }

    public DefaultHttp2FrameReader(boolean bl) {
        this(new DefaultHttp2HeadersDecoder(bl));
    }

    public DefaultHttp2FrameReader(Http2HeadersDecoder http2HeadersDecoder) {
        this.headersDecoder = http2HeadersDecoder;
        this.maxFrameSize = 16384;
    }

    @Override
    public Http2HeadersDecoder.Configuration headersConfiguration() {
        return this.headersDecoder.configuration();
    }

    @Override
    public Http2FrameReader.Configuration configuration() {
        return this;
    }

    @Override
    public Http2FrameSizePolicy frameSizePolicy() {
        return this;
    }

    @Override
    public void maxFrameSize(int n) {
        if (!Http2CodecUtil.isMaxFrameSizeValid(n)) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Invalid MAX_FRAME_SIZE specified in sent settings: %d", n);
        }
        this.maxFrameSize = n;
    }

    @Override
    public int maxFrameSize() {
        return this.maxFrameSize;
    }

    @Override
    public void close() {
        this.closeHeadersContinuation();
    }

    private void closeHeadersContinuation() {
        if (this.headersContinuation != null) {
            this.headersContinuation.close();
            this.headersContinuation = null;
        }
    }

    @Override
    public void readFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        if (this.readError) {
            ByteBuf byteBuf2 = byteBuf;
            byteBuf2.skipBytes(byteBuf2.readableBytes());
            return;
        }
        try {
            do {
                if (this.readingHeaders && !this.preProcessFrame(byteBuf)) {
                    return;
                }
                if (byteBuf.readableBytes() < this.payloadLength) {
                    return;
                }
                ByteBuf byteBuf3 = byteBuf.readSlice(this.payloadLength);
                this.readingHeaders = true;
                this.verifyFrameState();
                this.processPayloadState(channelHandlerContext, byteBuf3, http2FrameListener);
            } while (byteBuf.isReadable());
            return;
        }
        catch (Http2Exception http2Exception) {
            this.readError = !Http2Exception.isStreamError(http2Exception);
            throw http2Exception;
        }
        catch (RuntimeException runtimeException) {
            this.readError = true;
            throw runtimeException;
        }
        catch (Throwable throwable) {
            this.readError = true;
            PlatformDependent.throwException((Throwable)throwable);
            return;
        }
    }

    private boolean preProcessFrame(ByteBuf byteBuf) {
        if (byteBuf.readableBytes() < 9) {
            return false;
        }
        this.payloadLength = byteBuf.readUnsignedMedium();
        if (this.payloadLength > this.maxFrameSize) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length: %d exceeds maximum: %d", this.payloadLength, this.maxFrameSize);
        }
        this.frameType = byteBuf.readByte();
        this.flags = new Http2Flags(byteBuf.readUnsignedByte());
        this.streamId = Http2CodecUtil.readUnsignedInt(byteBuf);
        this.readingHeaders = false;
        return true;
    }

    private void verifyFrameState() {
        switch (this.frameType) {
            case 0: {
                this.verifyDataFrame();
                return;
            }
            case 1: {
                this.verifyHeadersFrame();
                return;
            }
            case 2: {
                this.verifyPriorityFrame();
                return;
            }
            case 3: {
                this.verifyRstStreamFrame();
                return;
            }
            case 4: {
                this.verifySettingsFrame();
                return;
            }
            case 5: {
                this.verifyPushPromiseFrame();
                return;
            }
            case 6: {
                this.verifyPingFrame();
                return;
            }
            case 7: {
                this.verifyGoAwayFrame();
                return;
            }
            case 8: {
                this.verifyWindowUpdateFrame();
                return;
            }
            case 9: {
                this.verifyContinuationFrame();
                return;
            }
        }
        this.verifyUnknownFrame();
    }

    private void processPayloadState(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        assert (byteBuf.readableBytes() == this.payloadLength);
        switch (this.frameType) {
            case 0: {
                this.readDataFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 1: {
                this.readHeadersFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 2: {
                this.readPriorityFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 3: {
                this.readRstStreamFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 4: {
                this.readSettingsFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 5: {
                this.readPushPromiseFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 6: {
                this.readPingFrame(channelHandlerContext, byteBuf.readLong(), http2FrameListener);
                return;
            }
            case 7: {
                this.readGoAwayFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 8: {
                this.readWindowUpdateFrame(channelHandlerContext, byteBuf, http2FrameListener);
                return;
            }
            case 9: {
                this.readContinuationFrame(byteBuf, http2FrameListener);
                return;
            }
        }
        this.readUnknownFrame(channelHandlerContext, byteBuf, http2FrameListener);
    }

    private void verifyDataFrame() {
        this.verifyAssociatedWithAStream();
        this.verifyNotProcessingHeaders();
        if (this.payloadLength < this.flags.getPaddingPresenceFieldLength()) {
            throw Http2Exception.streamError(this.streamId, Http2Error.FRAME_SIZE_ERROR, "Frame length %d too small.", this.payloadLength);
        }
    }

    private void verifyHeadersFrame() {
        this.verifyAssociatedWithAStream();
        this.verifyNotProcessingHeaders();
        int n = this.flags.getPaddingPresenceFieldLength() + this.flags.getNumPriorityBytes();
        if (this.payloadLength < n) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length %d too small for HEADERS frame with stream %d.", this.payloadLength, this.streamId);
        }
    }

    private void verifyPriorityFrame() {
        this.verifyAssociatedWithAStream();
        this.verifyNotProcessingHeaders();
        if (this.payloadLength != 5) {
            throw Http2Exception.streamError(this.streamId, Http2Error.FRAME_SIZE_ERROR, "Invalid frame length %d.", this.payloadLength);
        }
    }

    private void verifyRstStreamFrame() {
        this.verifyAssociatedWithAStream();
        this.verifyNotProcessingHeaders();
        if (this.payloadLength != 4) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Invalid frame length %d.", this.payloadLength);
        }
    }

    private void verifySettingsFrame() {
        this.verifyNotProcessingHeaders();
        if (this.streamId != 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "A stream ID must be zero.", new Object[0]);
        }
        if (this.flags.ack() && this.payloadLength > 0) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Ack settings frame must have an empty payload.", new Object[0]);
        }
        if (this.payloadLength % 6 > 0) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length %d invalid.", this.payloadLength);
        }
    }

    private void verifyPushPromiseFrame() {
        this.verifyNotProcessingHeaders();
        int n = this.flags.getPaddingPresenceFieldLength() + 4;
        if (this.payloadLength < n) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length %d too small for PUSH_PROMISE frame with stream id %d.", this.payloadLength, this.streamId);
        }
    }

    private void verifyPingFrame() {
        this.verifyNotProcessingHeaders();
        if (this.streamId != 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "A stream ID must be zero.", new Object[0]);
        }
        if (this.payloadLength != 8) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length %d incorrect size for ping.", this.payloadLength);
        }
    }

    private void verifyGoAwayFrame() {
        this.verifyNotProcessingHeaders();
        if (this.streamId != 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "A stream ID must be zero.", new Object[0]);
        }
        if (this.payloadLength < 8) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Frame length %d too small.", this.payloadLength);
        }
    }

    private void verifyWindowUpdateFrame() {
        this.verifyNotProcessingHeaders();
        DefaultHttp2FrameReader.verifyStreamOrConnectionId(this.streamId, "Stream ID");
        if (this.payloadLength != 4) {
            throw Http2Exception.connectionError(Http2Error.FRAME_SIZE_ERROR, "Invalid frame length %d.", this.payloadLength);
        }
    }

    private void verifyContinuationFrame() {
        this.verifyAssociatedWithAStream();
        if (this.headersContinuation == null) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Received %s frame but not currently processing headers.", this.frameType);
        }
        if (this.streamId != this.headersContinuation.getStreamId()) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Continuation stream ID does not match pending headers. Expected %d, but received %d.", this.headersContinuation.getStreamId(), this.streamId);
        }
    }

    private void verifyUnknownFrame() {
        this.verifyNotProcessingHeaders();
    }

    private void readDataFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        int n = this.readPadding(byteBuf);
        this.verifyPadding(n);
        int n2 = DefaultHttp2FrameReader.lengthWithoutTrailingPadding(byteBuf.readableBytes(), n);
        ByteBuf byteBuf2 = byteBuf;
        byteBuf2.writerIndex(byteBuf2.readerIndex() + n2);
        http2FrameListener.onDataRead(channelHandlerContext, this.streamId, byteBuf, n, this.flags.endOfStream());
    }

    private void readHeadersFrame(final ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        final int n = this.streamId;
        final Http2Flags http2Flags = this.flags;
        final int n2 = this.readPadding(byteBuf);
        this.verifyPadding(n2);
        if (this.flags.priorityPresent()) {
            long l = byteBuf.readUnsignedInt();
            final boolean bl = (l & 0x80000000L) != 0L;
            final int n3 = (int)(l & Integer.MAX_VALUE);
            if (n3 == this.streamId) {
                throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "HEADERS frame for stream %d cannot depend on itself.", this.streamId);
            }
            final short s = (short)(byteBuf.readUnsignedByte() + 1);
            int n4 = DefaultHttp2FrameReader.lengthWithoutTrailingPadding(byteBuf.readableBytes(), n2);
            this.headersContinuation = new HeadersContinuation(){

                @Override
                public int getStreamId() {
                    return n;
                }

                @Override
                public void processFragment(boolean bl2, ByteBuf byteBuf, int n4, Http2FrameListener http2FrameListener) {
                    HeadersBlockBuilder headersBlockBuilder = this.headersBlockBuilder();
                    headersBlockBuilder.addFragment(byteBuf, n4, channelHandlerContext.alloc(), bl2);
                    if (bl2) {
                        http2FrameListener.onHeadersRead(channelHandlerContext, n, headersBlockBuilder.headers(), n3, s, bl, n2, http2Flags.endOfStream());
                    }
                }
            };
            this.headersContinuation.processFragment(this.flags.endOfHeaders(), byteBuf, n4, http2FrameListener);
            DefaultHttp2FrameReader defaultHttp2FrameReader = this;
            defaultHttp2FrameReader.resetHeadersContinuationIfEnd(defaultHttp2FrameReader.flags.endOfHeaders());
            return;
        }
        this.headersContinuation = new HeadersContinuation(){

            @Override
            public int getStreamId() {
                return n;
            }

            @Override
            public void processFragment(boolean bl, ByteBuf byteBuf, int n3, Http2FrameListener http2FrameListener) {
                HeadersBlockBuilder headersBlockBuilder = this.headersBlockBuilder();
                headersBlockBuilder.addFragment(byteBuf, n3, channelHandlerContext.alloc(), bl);
                if (bl) {
                    http2FrameListener.onHeadersRead(channelHandlerContext, n, headersBlockBuilder.headers(), n2, http2Flags.endOfStream());
                }
            }
        };
        int n5 = DefaultHttp2FrameReader.lengthWithoutTrailingPadding(byteBuf.readableBytes(), n2);
        this.headersContinuation.processFragment(this.flags.endOfHeaders(), byteBuf, n5, http2FrameListener);
        DefaultHttp2FrameReader defaultHttp2FrameReader = this;
        defaultHttp2FrameReader.resetHeadersContinuationIfEnd(defaultHttp2FrameReader.flags.endOfHeaders());
    }

    private void resetHeadersContinuationIfEnd(boolean bl) {
        if (bl) {
            this.closeHeadersContinuation();
        }
    }

    private void readPriorityFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        long l = byteBuf.readUnsignedInt();
        boolean bl = (l & 0x80000000L) != 0L;
        int n = (int)(l & Integer.MAX_VALUE);
        if (n == this.streamId) {
            throw Http2Exception.streamError(this.streamId, Http2Error.PROTOCOL_ERROR, "A stream cannot depend on itself.", new Object[0]);
        }
        short s = (short)(byteBuf.readUnsignedByte() + 1);
        http2FrameListener.onPriorityRead(channelHandlerContext, this.streamId, n, s, bl);
    }

    private void readRstStreamFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        long l = byteBuf.readUnsignedInt();
        http2FrameListener.onRstStreamRead(channelHandlerContext, this.streamId, l);
    }

    private void readSettingsFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        if (this.flags.ack()) {
            http2FrameListener.onSettingsAckRead(channelHandlerContext);
            return;
        }
        int n = this.payloadLength / 6;
        Http2Settings http2Settings = new Http2Settings();
        for (int i = 0; i < n; ++i) {
            char c = (char)byteBuf.readUnsignedShort();
            long l = byteBuf.readUnsignedInt();
            try {
                http2Settings.put(c, l);
                continue;
            }
            catch (IllegalArgumentException illegalArgumentException) {
                if (c == '\u0004') {
                    throw Http2Exception.connectionError(Http2Error.FLOW_CONTROL_ERROR, illegalArgumentException, "Failed setting initial window size: %s", illegalArgumentException.getMessage());
                }
                throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, illegalArgumentException, "Protocol error: %s", illegalArgumentException.getMessage());
            }
        }
        http2FrameListener.onSettingsRead(channelHandlerContext, http2Settings);
    }

    private void readPushPromiseFrame(final ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        final int n = this.streamId;
        final int n2 = this.readPadding(byteBuf);
        this.verifyPadding(n2);
        final int n3 = Http2CodecUtil.readUnsignedInt(byteBuf);
        this.headersContinuation = new HeadersContinuation(){

            @Override
            public int getStreamId() {
                return n;
            }

            @Override
            public void processFragment(boolean bl, ByteBuf byteBuf, int n4, Http2FrameListener http2FrameListener) {
                this.headersBlockBuilder().addFragment(byteBuf, n4, channelHandlerContext.alloc(), bl);
                if (bl) {
                    http2FrameListener.onPushPromiseRead(channelHandlerContext, n, n3, this.headersBlockBuilder().headers(), n2);
                }
            }
        };
        int n4 = DefaultHttp2FrameReader.lengthWithoutTrailingPadding(byteBuf.readableBytes(), n2);
        this.headersContinuation.processFragment(this.flags.endOfHeaders(), byteBuf, n4, http2FrameListener);
        DefaultHttp2FrameReader defaultHttp2FrameReader = this;
        defaultHttp2FrameReader.resetHeadersContinuationIfEnd(defaultHttp2FrameReader.flags.endOfHeaders());
    }

    private void readPingFrame(ChannelHandlerContext channelHandlerContext, long l, Http2FrameListener http2FrameListener) {
        if (this.flags.ack()) {
            http2FrameListener.onPingAckRead(channelHandlerContext, l);
            return;
        }
        http2FrameListener.onPingRead(channelHandlerContext, l);
    }

    private void readGoAwayFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        int n = Http2CodecUtil.readUnsignedInt(byteBuf);
        long l = byteBuf.readUnsignedInt();
        http2FrameListener.onGoAwayRead(channelHandlerContext, n, l, byteBuf);
    }

    private void readWindowUpdateFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        int n = Http2CodecUtil.readUnsignedInt(byteBuf);
        if (n == 0) {
            if (this.streamId == 0) {
                throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Received WINDOW_UPDATE with delta 0 for connection stream", new Object[0]);
            }
            throw Http2Exception.streamError(this.streamId, Http2Error.PROTOCOL_ERROR, "Received WINDOW_UPDATE with delta 0 for stream: %d", this.streamId);
        }
        http2FrameListener.onWindowUpdateRead(channelHandlerContext, this.streamId, n);
    }

    private void readContinuationFrame(ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        this.headersContinuation.processFragment(this.flags.endOfHeaders(), byteBuf, this.payloadLength, http2FrameListener);
        DefaultHttp2FrameReader defaultHttp2FrameReader = this;
        defaultHttp2FrameReader.resetHeadersContinuationIfEnd(defaultHttp2FrameReader.flags.endOfHeaders());
    }

    private void readUnknownFrame(ChannelHandlerContext channelHandlerContext, ByteBuf byteBuf, Http2FrameListener http2FrameListener) {
        http2FrameListener.onUnknownFrame(channelHandlerContext, this.frameType, this.streamId, this.flags, byteBuf);
    }

    private int readPadding(ByteBuf byteBuf) {
        if (!this.flags.paddingPresent()) {
            return 0;
        }
        return byteBuf.readUnsignedByte() + 1;
    }

    private void verifyPadding(int n) {
        if ((n = DefaultHttp2FrameReader.lengthWithoutTrailingPadding(this.payloadLength, n)) < 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Frame payload too small for padding.", new Object[0]);
        }
    }

    private static int lengthWithoutTrailingPadding(int n, int n2) {
        if (n2 == 0) {
            return n;
        }
        return n - (n2 - 1);
    }

    private void verifyNotProcessingHeaders() {
        if (this.headersContinuation != null) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Received frame of type %s while processing headers on stream %d.", this.frameType, this.headersContinuation.getStreamId());
        }
    }

    private void verifyAssociatedWithAStream() {
        if (this.streamId == 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "Frame of type %s must be associated with a stream.", this.frameType);
        }
    }

    private static void verifyStreamOrConnectionId(int n, String string) {
        if (n < 0) {
            throw Http2Exception.connectionError(Http2Error.PROTOCOL_ERROR, "%s must be >= 0", string);
        }
    }

    protected class HeadersBlockBuilder {
        private ByteBuf headerBlock;

        protected HeadersBlockBuilder() {
        }

        private void headerSizeExceeded() {
            this.close();
            Http2CodecUtil.headerListSizeExceeded(DefaultHttp2FrameReader.this.headersDecoder.configuration().maxHeaderListSizeGoAway());
        }

        final void addFragment(ByteBuf byteBuf, int n, ByteBufAllocator byteBufAllocator, boolean bl) {
            if (this.headerBlock == null) {
                if ((long)n > DefaultHttp2FrameReader.this.headersDecoder.configuration().maxHeaderListSizeGoAway()) {
                    this.headerSizeExceeded();
                }
                if (bl) {
                    this.headerBlock = byteBuf.readRetainedSlice(n);
                    return;
                }
                this.headerBlock = byteBufAllocator.buffer(n).writeBytes(byteBuf, n);
                return;
            }
            if (DefaultHttp2FrameReader.this.headersDecoder.configuration().maxHeaderListSizeGoAway() - (long)n < (long)this.headerBlock.readableBytes()) {
                this.headerSizeExceeded();
            }
            if (this.headerBlock.isWritable(n)) {
                this.headerBlock.writeBytes(byteBuf, n);
                return;
            }
            byteBufAllocator = byteBufAllocator.buffer(this.headerBlock.readableBytes() + n);
            byteBufAllocator.writeBytes(this.headerBlock).writeBytes(byteBuf, n);
            this.headerBlock.release();
            this.headerBlock = byteBufAllocator;
        }

        Http2Headers headers() {
            try {
                Http2Headers http2Headers = DefaultHttp2FrameReader.this.headersDecoder.decodeHeaders(DefaultHttp2FrameReader.this.streamId, this.headerBlock);
                return http2Headers;
            }
            finally {
                this.close();
            }
        }

        void close() {
            if (this.headerBlock != null) {
                this.headerBlock.release();
                this.headerBlock = null;
            }
            DefaultHttp2FrameReader.this.headersContinuation = null;
        }
    }

    private abstract class HeadersContinuation {
        private final HeadersBlockBuilder builder;

        private HeadersContinuation() {
            this.builder = new HeadersBlockBuilder();
        }

        abstract int getStreamId();

        abstract void processFragment(boolean var1, ByteBuf var2, int var3, Http2FrameListener var4);

        final HeadersBlockBuilder headersBlockBuilder() {
            return this.builder;
        }

        final void close() {
            this.builder.close();
        }
    }
}

